<?php

namespace app\esign\service;

/**
 * E签宝工具类
 * 
 * Class EsignHelper
 */
class EsignHelper {

    protected $api_url='';
    protected $host=array(
        'smlopenapi'=>'https://smlopenapi.esign.cn',    // 沙箱环境
        'openapi'=>'https://openapi.esign.cn'           // 生产环境
    );
    protected $env_host;
    protected $app_id;
    protected $app_secret;
    protected $data='';
    protected $headers=array();
    protected $http_helper;
    protected $method='GET';

    /**
     * EsignHelper constructor.
     * 
     * @access public
     * @param HttpHelper $http_helper http请求工具类
     * @param string $app_id 应用id
     * @param string $app_secret 应用密钥
     * @param string $api_url api地址
     * @param string $env 环境 (smlopenapi|openapi)
     * @return void
     */
    public function __construct(HttpHelper $http_helper,string $app_id,string $app_secret,string $api_url,string $env='smlopenapi'){
        $this->http_helper=$http_helper;
        $this->env_host=$this->host[$env]??$env;
        $this->app_id=$app_id;
        $this->api_url=$api_url;
        $this->app_secret=$app_secret;
        $this->setHeaders();
    }

    /**
     * 传入请求参数
     * 
     * @access public
     * @param array $data 请求参数
     * @return self
     */
    public function setData(array $data){
        $this->method=empty($data)?'GET':'POST';
        $this->data=json_encode($data);
        return $this->setHeaders();
    }

    /**
     * 发送请求
     * 
     * @access public
     * @return array
     */
    public function send(){
        $this->http_helper->setUrl($this->env_host.$this->api_url)
            ->setMethod($this->method)
            ->setHeaders($this->headers)
            ->setBody($this->data)
            ->execute();
        // 写入日志
        $log_path=app()->getRootPath().'runtime/log/request'.date('Ymd').'.log';
        $log=date('Y-m-d H:i:s').' '.json_encode(array(
            'url'=>$this->env_host.$this->api_url,
            'method'=>$this->method,
            'headers'=>$this->headers,
            'body'=>$this->data,
            'response'=>array(
                'status_code'=>$this->http_helper->getStatusCode(),
                'headers'=>$this->http_helper->getHeaders(),
                'body'=>$this->http_helper->getBody()
            )
        ),JSON_UNESCAPED_UNICODE)."\n";
        file_put_contents($log_path,$log,FILE_APPEND);
        return array(
            'status_code'=>$this->http_helper->getStatusCode(),
            'headers'=>$this->http_helper->getHeaders(),
            'body'=>$this->http_helper->getBody()
        );
    }

    /**
     * 计算请求头
     * 
     * @access public
     * @return self
     */
    public function setHeaders(){
        $content_md5=$this->getBodyContentMD5($this->data);
        $sign=array(
            'HTTPMethod'=>$this->method,
            'Accept'=>'*/*',
            'Content-MD5'=>$content_md5,
            'Content-Type'=>'application/json; charset=UTF-8',
            'Date'=>'',
            'Headers'=>'',
            'PathAndParameters'=>$this->api_url
        );
        $sign_str='';
        foreach ($sign as $key=>$value){
            if($key=='Headers'&&$value=='')
                continue;
            $sign_str.=(empty($sign_str)?'':"\n").$value;
        }
        $this->headers=array(
            'X-Tsign-Open-App-Id:'.$this->app_id,
            'X-Tsign-Open-Auth-Mode:Signature',
            'X-Tsign-Open-Ca-Signature:'.$this->sign($sign_str),
            'X-Tsign-Open-Ca-Timestamp:'.time()*1000,
            'Accept:*/*',
            'Content-Type:application/json; charset=UTF-8',
            'Content-MD5:'.$content_md5
        );
        return $this;
    }

    /**
     * 计算body的md5值
     * 
     * @access protected
     * @param string $bodyData body数据
     * @return string
     */
    protected function getBodyContentMD5($bodyData){
        $temp_file_name=md5(time().rand(1000,99999999));
        file_put_contents($temp_file_name,$bodyData);
        $md5Bytes=md5_file($temp_file_name,true);
        unlink($temp_file_name);
        $contentMD5=base64_encode($md5Bytes);
        return $contentMD5;
    }

    /**
     * 计算签名
     * 
     * @access protected
     * @param string $sign_str 待签名字符串
     * @return string
     */
    protected function sign($sign_str){
        $sign_str=hash_hmac("sha256",$sign_str,$this->app_secret,true);
        return base64_encode($sign_str);
    }

}

?>