<?php

namespace app\esign\service;

use app\esign\service\Config;
use app\esign\service\HttpHelper;
use app\esign\service\EsignHelper;

/**
 * E签宝合同类(零时,暂不考虑耦合性)
 * 
 * Class Contract
 * 
 * @document https://open.esign.cn/doc/opendoc/pdf-sign3/su5g42 基于文件发起签署
 * @document https://open.esign.cn/doc/opendoc/file-and-template3/mv8a3i 填写模板生成文件
 * @document https://open.esign.cn/doc/opendoc/pdf-sign3/pvfkwd 获取签署页面链接
 * @document https://open.esign.cn/doc/opendoc/pdf-sign3/ynwqsm 完结签署流程
 * @document https://open.esign.cn/doc/opendoc/pdf-sign3/klbicu 撤销签署流程(签署中的流程)
 * @document https://open.esign.cn/doc/opendoc/pdf-sign3/ohzup7 追加签署区(自动落章)
 */

class Contract {

    public $http_helper;
    public $data;
    public $sign_flow_id='';
    public $file_id='';
    public $app_id='';
    public $app_secret='';
    public $env='';

    /**
     * Contract constructor.
     * 
     * @access public
     * @return void
     */
    public function __construct(){
        $this->http_helper=new HttpHelper();
        $this->app_id=Config::get('app_id');
        $this->app_secret=Config::get('app_secret');
        $this->env=Config::get('env');
    }

    /**
     * 请求助手方法
     * 
     * @access protected
     * @param string $api_url api地址
     * @param array $data 请求参数
     * @throws \Exception
     * @return array
     */
    protected function request(string $api_url,array $data=[]){
        $esign_helper=new EsignHelper($this->http_helper,$this->app_id,$this->app_secret,$api_url,$this->env);
        $this->data=$esign_helper->setData($data)->send();
        $data=json_decode($this->data['body'],true);
        if($data['code']!==0) throw new \Exception($data['message']);
        return $data;
    }

    /**
     * 填充模板信息
     * 
     * @access public
     * @param array $data
     * @throws \Exception
     * @return string
     */
    public function fill(array $data){
        $api_url='/v3/files/create-by-doc-template';
        $data=$this->request($api_url,$data);
        $this->file_id=$data['data']['fileId'];
        return $this->file_id;
    }

    /**
     * 创建签署流程
     * 
     * @access public
     * @param array $data 请求参数
     * @throws \Exception
     * @return self
     */
    public function create(array $data) {
        $api_url='/v3/sign-flow/create-by-file';
        $data=$this->request($api_url,$data);
        $this->sign_flow_id=$data['data']['signFlowId'];
        return $this;
    }

    /**
     * 获取签署地址
     * 
     * @access public
     * @param string $sign_flow_id 签署流程id
     * @param string $phone 签署人手机号(为空则为签署流程发起人)
     * @throws \Exception
     * @return string
     */
    public function url(?string $sign_flow_id=null,?string $phone=null) {
        $sign_flow_id=$sign_flow_id??$this->sign_flow_id;
        if(empty($sign_flow_id)) throw new \Exception('签署流程id不能为空');
        $api_url="/v3/sign-flow/{$sign_flow_id}/sign-url";
        $data=array(
            'signFlowId'=>$sign_flow_id
        );
        if($phone!==null)
            $data['operator']=array(
                'psnAccount'=>$phone
            );
        $data=$this->request($api_url,$data);
        return $data['data']['shortUrl'];
    }

    /**
     * 自动落章
     * 
     * @access public
     * @param array $data 请求参数
     * @param string $sign_flow_id 签署流程id
     * @throws \Exception
     * @return self
     */
    public function sign(array $data,?string $sign_flow_id=null) {
        $sign_flow_id=$sign_flow_id??$this->sign_flow_id;
        if(empty($sign_flow_id)) throw new \Exception('签署流程id不能为空');
        $api_url="/v3/sign-flow/{$sign_flow_id}/signers/sign-fields";
        $data=$this->request($api_url,$data);
        return $this;
    }

    /**
     * 结束签署流程
     * 
     * @access public
     * @param string $sign_flow_id 签署流程id
     * @throws \Exception
     * @return self
     */
    public function finish(?string $sign_flow_id=null) {
        $sign_flow_id=$sign_flow_id??$this->sign_flow_id;
        if(empty($sign_flow_id)) throw new \Exception('签署流程id不能为空');
        $api_url="/v3/sign-flow/{$sign_flow_id}/finish";
        $this->request($api_url,array(
            'signFlowId'=>$sign_flow_id
        ));
        return $this;
    }

    /**
     * 撤销签署流程(签署中的流程)
     * 
     * @access public
     * @param string $sign_flow_id 签署流程id
     * @param string $revoke_reason 撤销原因
     * @throws \Exception
     * @return self
     */
    public function cancel(?string $sign_flow_id=null,?string $revoke_reason=null) {
        $sign_flow_id=$sign_flow_id??$this->sign_flow_id;
        if(empty($sign_flow_id)) throw new \Exception('签署流程id不能为空');
        $api_url="/v3/sign-flow/{$sign_flow_id}/revoke";
        $this->request($api_url,array(
            'signFlowId'=>$sign_flow_id,
            'revokeReason'=>$revoke_reason??'无'
        ));
        return $this;
    }


    /**
     * 将数字转换为中文大写
     * 
     * @access public
     * @param float $num 数字
     * @return string
     */
    public static function numtocn(float $num){
        $cn_number=array('零','壹','贰','叁','肆','伍','陆','柒','捌','玖');
        $cn_measurement=array('分','角','元','拾','佰','仟','万','拾万','佰万','仟万','亿','拾亿','佰亿','仟亿');
        $num=round($num,2);
        $num*=100;
        if(strlen($num)>14) return '数字太大,无法转换';
        $num_str=strrev($num);
        $cn_str='';
        $len=strlen($num_str);
        for($i=0;$i<$len;$i++){
            $temp_str=$cn_number[$num_str[$i]]??'';
            if($temp_str=='零'){
                // 判断上一个字符是否是零,如果是零,则不添加
                $netx_str=$cn_number[$num_str[$i-1]]??'';
                if($netx_str!='零') $cn_str=$temp_str.$cn_str;
            } else {
                $cn_str=$temp_str.$cn_measurement[$i].$cn_str;
            }
        }
        // 通过正则表达式去除末尾的零
        $cn_str=preg_replace('/零.?$/u','',$cn_str);
        // 如果末尾不算以“分”“角”“元”结尾,则添加“元”
        if(!preg_match('/分$|角$|元$/u',$cn_str)) $cn_str.='元';
        // 如果末尾是以“元”结尾,则添加“整”
        if(preg_match('/元$/u',$cn_str)) $cn_str.='整';
        return $cn_str;
    }

}