<?php

namespace app\esign\service;

use app\esign\service\Config;
use app\esign\service\HttpHelper;
use app\esign\service\EsignHelper;

/**
 * E签宝公司类
 * 
 * Class Company
 * 
 * @document https://open.esign.cn/doc/opendoc/seal3/ups6h1 查询企业内部印章
 * @document https://open.esign.cn/doc/opendoc/seal3/czrua1 查询被外部企业授权印章
 */
class Company {

    public $http_helper;
    public $data;
    public $sign_flow_id='';
    public $file_id='';
    public $app_id='';
    public $app_secret='';
    public $env='';
    public $org_id='';

    /**
     * Company constructor.
     * 
     * @access public
     * @return void
     */
    public function __construct(){
        $this->http_helper=new HttpHelper();
        $this->app_id=Config::get('app_id');
        $this->app_secret=Config::get('app_secret');
        $this->env=Config::get('env');
        $this->org_id=Config::get('org_id');
    }

    /**
     * 请求助手方法
     * 
     * @access protected
     * @param string $api_url api地址
     * @param array $data 请求参数
     * @throws \Exception
     * @return array
     */
    protected function request(string $api_url,array $data=[]){
        $esign_helper=new EsignHelper($this->http_helper,$this->app_id,$this->app_secret,$api_url,$this->env);
        $this->data=$esign_helper->setData($data)->send();
        $data=json_decode($this->data['body'],true);
        if($data['code']!==0) throw new \Exception($data['message']);
        return $data;
    }

    /**
     * 获取本公司的所有印章
     * 
     * @access public
     * @param string $type 印章类型
     * @param int $page_num 页码
     * @param int $page_size 每页数量
     * @throws \Exception
     * @return array
     */
    public function getSeals(?string $type=null,int $page_num=1,int $page_size=20){
        $api_url="/v3/seals/org-own-seal-list?orgId={$this->org_id}&pageNum={$page_num}&pageSize={$page_size}";
        if($type) $api_url.="&sealBizTypes={$type}";
        return $this->request($api_url)['data'];
    }

    /**
     * 查询本公司获得授权的所有印章
     * 
     * @access public
     * @param int $page_num 页码
     * @param int $page_size 每页数量
     * @param string $authorizer_org_id 授权者机构ID
     * @throws \Exception
     * @return array
     */
    public function getAuthorizedSeals(int $page_num=1,int $page_size=20,?string $authorizer_org_id=null){
        $api_url="/v3/seals/org-authorized-seal-list?orgId={$this->org_id}&pageNum={$page_num}&pageSize={$page_size}";
        if($authorizer_org_id) $api_url.="&authorizerOrgId={$authorizer_org_id}";
        return $this->request($api_url)['data'];
    }

}